<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$user = $conn->query("SELECT * FROM users WHERE id = $user_id")->fetch_assoc();

// Get deposit methods from admin settings
$deposit_methods = [
    [
        'id' => 'bkash',
        'name' => 'bKash',
        'icon' => 'fas fa-mobile-alt',
        'color' => 'bg-pink-100 text-pink-600',
        'number' => '০১৭১২৩৪৫৬৭৮',
        'type' => 'Personal'
    ],
    [
        'id' => 'nagad',
        'name' => 'Nagad',
        'icon' => 'fas fa-wallet',
        'color' => 'bg-red-100 text-red-600',
        'number' => '০১৮১২৩৪৫৬৭৮',
        'type' => 'Personal'
    ],
    [
        'id' => 'rocket',
        'name' => 'Rocket',
        'icon' => 'fas fa-rocket',
        'color' => 'bg-purple-100 text-purple-600',
        'number' => '০১৯১২৩৪৫৬৭৮',
        'type' => 'Personal'
    ],
    [
        'id' => 'bank',
        'name' => 'Bank Transfer',
        'icon' => 'fas fa-university',
        'color' => 'bg-blue-100 text-blue-600',
        'details' => 'Bank: DBBL<br>Account: 123456789<br>Branch: Dhaka'
    ]
];

// Handle deposit request
if($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_deposit'])) {
    $amount = (float)$_POST['amount'];
    $method = sanitize($_POST['method']);
    $sender_number = sanitize($_POST['sender_number']);
    $transaction_id = sanitize($_POST['transaction_id']);
    
    if($amount < 100) {
        $error = "Minimum deposit amount is ৳100";
    } elseif($amount > 50000) {
        $error = "Maximum deposit amount is ৳50,000";
    } elseif(empty($sender_number)) {
        $error = "Sender number is required";
    } elseif(empty($transaction_id)) {
        $error = "Transaction ID is required";
    } else {
        // Check if transaction ID already exists
        $check_stmt = $conn->prepare("SELECT id FROM transactions WHERE description LIKE ? AND status = 'pending'");
        $search_term = "%$transaction_id%";
        $check_stmt->bind_param("s", $search_term);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if($check_result->num_rows > 0) {
            $error = "This transaction ID is already submitted";
        } else {
            // Create deposit request
            $stmt = $conn->prepare("INSERT INTO transactions (user_id, type, amount, description, status) VALUES (?, 'deposit', ?, ?, 'pending')");
            $description = "Deposit via $method - Sender: $sender_number - Txn: $transaction_id";
            $stmt->bind_param("ids", $user_id, $amount, $description);
            
            if($stmt->execute()) {
                $success = "Deposit request submitted successfully! It will be processed within 30 minutes.";
                // Send notification to admin
                $admin_notification = "New deposit request from {$user['username']} for ৳$amount via $method";
                $conn->query("INSERT INTO admin_notifications (message, type) VALUES ('$admin_notification', 'deposit')");
            } else {
                $error = "Failed to submit deposit request";
            }
        }
    }
}

// Get pending deposits
$pending_deposits = $conn->query("
    SELECT * FROM transactions 
    WHERE user_id = $user_id AND type = 'deposit' AND status = 'pending'
    ORDER BY created_at DESC
");

// Get recent deposits
$recent_deposits = $conn->query("
    SELECT * FROM transactions 
    WHERE user_id = $user_id AND type = 'deposit' 
    ORDER BY created_at DESC 
    LIMIT 5
");
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Deposit - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .method-card {
            transition: all 0.3s ease;
        }
        .method-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
        }
        .method-card.selected {
            border-color: #3B82F6;
            border-width: 2px;
        }
        .amount-option {
            transition: all 0.2s ease;
        }
        .amount-option:hover {
            transform: scale(1.05);
        }
        .amount-option.selected {
            transform: scale(1.05);
            border-color: #3B82F6;
            background-color: #EFF6FF;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="dashboard.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Dashboard</a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">Deposit</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-plus-circle mr-2"></i> Deposit Funds
                </h1>
                <p class="text-gray-600">Add money to your wallet balance</p>
            </div>
            
            <!-- Messages -->
            <?php if(isset($error)): ?>
                <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-lg animate__animated animate__shakeX">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-3 text-xl"></i>
                        <span class="text-red-700 font-medium"><?php echo $error; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if(isset($success)): ?>
                <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg animate__animated animate__fadeIn">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-xl"></i>
                        <span class="text-green-700 font-medium"><?php echo $success; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Left Column - Deposit Form -->
                <div class="lg:col-span-2 space-y-8">
                    <!-- Balance Card -->
                    <div class="bg-gradient-to-r from-blue-500 to-indigo-600 rounded-2xl shadow-lg p-8 text-white animate__animated animate__fadeInLeft">
                        <div class="flex flex-col md:flex-row items-center justify-between">
                            <div>
                                <h2 class="text-2xl font-bold mb-2">Current Balance</h2>
                                <p class="text-blue-100 opacity-90">Available for shopping and withdrawals</p>
                            </div>
                            <div class="mt-4 md:mt-0 text-center md:text-right">
                                <div class="text-5xl font-bold mb-2">৳<?php echo number_format($user['balance'], 2); ?></div>
                                <a href="withdraw.php" class="text-blue-100 hover:text-white underline">
                                    <i class="fas fa-arrow-right mr-1"></i> Want to withdraw?
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Deposit Methods -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-green-500 to-emerald-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-credit-card mr-2"></i> Select Payment Method
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                                <?php foreach($deposit_methods as $method): ?>
                                <div class="method-card cursor-pointer border-2 border-gray-200 rounded-xl p-4 text-center hover:border-blue-300 transition duration-300"
                                     onclick="selectMethod('<?php echo $method['id']; ?>')"
                                     id="method-<?php echo $method['id']; ?>">
                                    <div class="w-12 h-12 <?php echo $method['color']; ?> rounded-full flex items-center justify-center mx-auto mb-3">
                                        <i class="<?php echo $method['icon']; ?> text-xl"></i>
                                    </div>
                                    <h4 class="font-bold text-gray-800"><?php echo $method['name']; ?></h4>
                                    <p class="text-gray-600 text-sm mt-1"><?php echo $method['type'] ?? 'Transfer'; ?></p>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <!-- Method Details -->
                            <div id="methodDetails" class="hidden">
                                <div class="bg-gray-50 rounded-xl p-6 mb-6">
                                    <div id="methodContent"></div>
                                </div>
                                
                                <!-- Amount Selection -->
                                <div class="mb-6">
                                    <h4 class="text-lg font-bold text-gray-800 mb-4">
                                        <i class="fas fa-coins mr-2"></i> Select Amount
                                    </h4>
                                    <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-5 gap-3 mb-6">
                                        <?php 
                                        $amounts = [100, 200, 500, 1000, 2000, 3000, 5000, 10000, 20000, 50000];
                                        foreach($amounts as $amount): 
                                        ?>
                                        <button type="button" 
                                                onclick="selectAmount(<?php echo $amount; ?>)"
                                                class="amount-option border-2 border-gray-200 rounded-lg p-4 text-center hover:border-blue-300 transition duration-300"
                                                id="amount-<?php echo $amount; ?>">
                                            <span class="text-lg font-bold text-gray-800">৳<?php echo number_format($amount); ?></span>
                                        </button>
                                        <?php endforeach; ?>
                                    </div>
                                    
                                    <!-- Custom Amount -->
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            Or Enter Custom Amount
                                        </label>
                                        <div class="relative">
                                            <span class="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-500 font-bold">৳</span>
                                            <input type="number" 
                                                   id="customAmount"
                                                   name="amount"
                                                   min="100" 
                                                   max="50000"
                                                   step="10"
                                                   class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                   placeholder="Enter amount (100 - 50000)"
                                                   oninput="selectCustomAmount(this.value)">
                                        </div>
                                        <p class="text-gray-500 text-xs mt-2">
                                            Minimum: ৳100, Maximum: ৳50,000
                                        </p>
                                    </div>
                                </div>
                                
                                <!-- Deposit Form -->
                                <form method="POST" action="" id="depositForm">
                                    <input type="hidden" name="method" id="selectedMethod">
                                    <input type="hidden" name="amount" id="selectedAmount">
                                    
                                    <div class="space-y-4">
                                        <div>
                                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                                Your Sending Number
                                            </label>
                                            <input type="text" 
                                                   name="sender_number"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                   placeholder="01XXXXXXXXX"
                                                   required>
                                        </div>
                                        
                                        <div>
                                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                                Transaction ID
                                            </label>
                                            <input type="text" 
                                                   name="transaction_id"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                   placeholder="Enter transaction ID"
                                                   required>
                                            <p class="text-gray-500 text-xs mt-2">
                                                Find this in your payment confirmation message
                                            </p>
                                        </div>
                                        
                                        <!-- Instructions -->
                                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                            <h5 class="font-bold text-blue-800 mb-2">
                                                <i class="fas fa-info-circle mr-2"></i> Important Instructions
                                            </h5>
                                            <ul class="list-disc list-inside text-blue-700 text-sm space-y-1">
                                                <li>Send exact amount as selected</li>
                                                <li>Keep transaction ID for verification</li>
                                                <li>Deposit will be processed within 30 minutes</li>
                                                <li>Contact support if not processed within 1 hour</li>
                                            </ul>
                                        </div>
                                        
                                        <!-- Submit Button -->
                                        <div class="pt-4">
                                            <button type="submit" name="submit_deposit"
                                                    class="w-full bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-4 px-6 rounded-lg text-lg hover:from-green-600 hover:to-emerald-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105">
                                                <i class="fas fa-paper-plane mr-2"></i> Submit Deposit Request
                                            </button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <!-- How to Deposit -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft" style="animation-delay: 0.2s">
                        <div class="bg-gradient-to-r from-purple-500 to-pink-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-question-circle mr-2"></i> How to Deposit
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="space-y-6">
                                <div class="flex items-start">
                                    <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-blue-600 font-bold">1</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 mb-2">Select Payment Method</h4>
                                        <p class="text-gray-600">Choose from bKash, Nagad, Rocket or Bank Transfer</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start">
                                    <div class="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-green-600 font-bold">2</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 mb-2">Send Payment</h4>
                                        <p class="text-gray-600">Send money to the provided number/account</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start">
                                    <div class="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-yellow-600 font-bold">3</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 mb-2">Submit Details</h4>
                                        <p class="text-gray-600">Enter your sending number and transaction ID</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start">
                                    <div class="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-purple-600 font-bold">4</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 mb-2">Wait for Approval</h4>
                                        <p class="text-gray-600">Admin will verify and add balance within 30 minutes</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Right Column - Pending & Recent Deposits -->
                <div class="space-y-8">
                    <!-- Pending Deposits -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight">
                        <div class="bg-gradient-to-r from-yellow-500 to-orange-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-clock mr-2"></i> Pending Deposits
                            </h3>
                        </div>
                        <div class="p-6">
                            <?php if($pending_deposits->num_rows > 0): ?>
                                <div class="space-y-4">
                                    <?php while($deposit = $pending_deposits->fetch_assoc()): ?>
                                    <div class="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                                        <div class="flex justify-between items-start mb-2">
                                            <span class="font-bold text-gray-800">
                                                ৳<?php echo number_format($deposit['amount'], 2); ?>
                                            </span>
                                            <span class="px-2 py-1 bg-yellow-100 text-yellow-800 text-xs font-bold rounded">
                                                Pending
                                            </span>
                                        </div>
                                        <p class="text-gray-600 text-sm mb-2">
                                            <?php echo $deposit['description']; ?>
                                        </p>
                                        <p class="text-gray-500 text-xs">
                                            <i class="far fa-clock mr-1"></i>
                                            <?php echo date('M d, h:i A', strtotime($deposit['created_at'])); ?>
                                        </p>
                                    </div>
                                    <?php endwhile; ?>
                                </div>
                            <?php else: ?>
                                <div class="text-center py-8">
                                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                        <i class="fas fa-check text-gray-400 text-2xl"></i>
                                    </div>
                                    <p class="text-gray-600">No pending deposits</p>
                                </div>
                            <?php endif; ?>
                            
                            <div class="mt-6">
                                <a href="transactions.php?type=deposit" 
                                   class="block text-center text-blue-600 hover:text-blue-800 font-medium">
                                    <i class="fas fa-list mr-1"></i> View All Transactions
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Recent Deposits -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-blue-500 to-indigo-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-history mr-2"></i> Recent Deposits
                            </h3>
                        </div>
                        <div class="p-6">
                            <?php if($recent_deposits->num_rows > 0): ?>
                                <div class="space-y-4">
                                    <?php while($deposit = $recent_deposits->fetch_assoc()): 
                                        $status_color = $deposit['status'] == 'completed' ? 'green' : 
                                                       ($deposit['status'] == 'pending' ? 'yellow' : 'red');
                                    ?>
                                    <div class="p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition duration-300">
                                        <div class="flex justify-between items-start mb-2">
                                            <span class="font-bold text-gray-800">
                                                ৳<?php echo number_format($deposit['amount'], 2); ?>
                                            </span>
                                            <span class="px-2 py-1 bg-<?php echo $status_color; ?>-100 text-<?php echo $status_color; ?>-800 text-xs font-bold rounded">
                                                <?php echo ucfirst($deposit['status']); ?>
                                            </span>
                                        </div>
                                        <p class="text-gray-600 text-sm mb-2 truncate">
                                            <?php echo substr($deposit['description'], 0, 50); ?>...
                                        </p>
                                        <p class="text-gray-500 text-xs">
                                            <?php echo date('M d, h:i A', strtotime($deposit['created_at'])); ?>
                                        </p>
                                    </div>
                                    <?php endwhile; ?>
                                </div>
                            <?php else: ?>
                                <div class="text-center py-8">
                                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                        <i class="fas fa-history text-gray-400 text-2xl"></i>
                                    </div>
                                    <p class="text-gray-600">No deposit history</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Support Card -->
                    <div class="bg-gradient-to-r from-green-500 to-emerald-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInRight" style="animation-delay: 0.2s">
                        <h3 class="text-xl font-bold mb-4">
                            <i class="fas fa-headset mr-2"></i> Need Help?
                        </h3>
                        <p class="mb-6 opacity-90">
                            Having trouble with deposit? Contact our support team.
                        </p>
                        <div class="space-y-3">
                            <a href="https://t.me/support" target="_blank" 
                               class="flex items-center justify-center bg-white text-blue-600 font-bold py-3 px-4 rounded-lg hover:bg-blue-50 transition duration-300">
                                <i class="fab fa-telegram mr-2 text-xl"></i> Telegram Support
                            </a>
                            <a href="https://wa.me/8801234567890" target="_blank" 
                               class="flex items-center justify-center bg-white text-green-600 font-bold py-3 px-4 rounded-lg hover:bg-green-50 transition duration-300">
                                <i class="fab fa-whatsapp mr-2 text-xl"></i> WhatsApp Support
                            </a>
                        </div>
                    </div>
                    
                    <!-- Quick Links -->
                    <div class="bg-white rounded-xl shadow-lg p-6 animate__animated animate__fadeInRight" style="animation-delay: 0.3s">
                        <h3 class="font-bold text-gray-800 mb-4">
                            <i class="fas fa-bolt mr-2"></i> Quick Actions
                        </h3>
                        <div class="space-y-3">
                            <a href="withdraw.php" 
                               class="flex items-center justify-between p-3 bg-gray-50 hover:bg-red-50 hover:text-red-600 rounded-lg transition duration-300 group">
                                <div class="flex items-center">
                                    <i class="fas fa-minus-circle text-gray-400 group-hover:text-red-500 mr-3"></i>
                                    <span>Withdraw Funds</span>
                                </div>
                                <i class="fas fa-chevron-right text-gray-300 group-hover:text-red-500"></i>
                            </a>
                            
                            <a href="transactions.php" 
                               class="flex items-center justify-between p-3 bg-gray-50 hover:bg-blue-50 hover:text-blue-600 rounded-lg transition duration-300 group">
                                <div class="flex items-center">
                                    <i class="fas fa-exchange-alt text-gray-400 group-hover:text-blue-500 mr-3"></i>
                                    <span>Transaction History</span>
                                </div>
                                <i class="fas fa-chevron-right text-gray-300 group-hover:text-blue-500"></i>
                            </a>
                            
                            <a href="dashboard.php" 
                               class="flex items-center justify-between p-3 bg-gray-50 hover:bg-green-50 hover:text-green-600 rounded-lg transition duration-300 group">
                                <div class="flex items-center">
                                    <i class="fas fa-tachometer-alt text-gray-400 group-hover:text-green-500 mr-3"></i>
                                    <span>Dashboard</span>
                                </div>
                                <i class="fas fa-chevron-right text-gray-300 group-hover:text-green-500"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Method selection
    let selectedMethod = '';
    let selectedAmount = 0;
    
    function selectMethod(methodId) {
        selectedMethod = methodId;
        
        // Update UI
        document.querySelectorAll('.method-card').forEach(card => {
            card.classList.remove('selected');
        });
        document.getElementById('method-' + methodId).classList.add('selected');
        
        // Show method details
        document.getElementById('methodDetails').classList.remove('hidden');
        document.getElementById('selectedMethod').value = methodId;
        
        // Update method content
        const methods = {
            'bkash': {
                name: 'bKash',
                number: '০১৭১২৩৪৫৬৭৮',
                instructions: 'Send money to this bKash number and enter transaction details below.'
            },
            'nagad': {
                name: 'Nagad',
                number: '০১৮১২৩৪৫৬৭৮',
                instructions: 'Send money to this Nagad number and enter transaction details below.'
            },
            'rocket': {
                name: 'Rocket',
                number: '০১৯১২৩৪৫৬৭৮',
                instructions: 'Send money to this Rocket number and enter transaction details below.'
            },
            'bank': {
                name: 'Bank Transfer',
                details: 'Bank: DBBL<br>Account: 123456789<br>Branch: Dhaka<br>Routing: 123456',
                instructions: 'Transfer money to this bank account and enter transaction details below.'
            }
        };
        
        const method = methods[methodId];
        let content = `
            <h4 class="text-xl font-bold text-gray-800 mb-2">${method.name}</h4>
            ${method.number ? `<p class="text-gray-700 mb-2"><strong>Number:</strong> ${method.number}</p>` : ''}
            ${method.details ? `<div class="text-gray-700 mb-2"><strong>Account Details:</strong><br>${method.details}</div>` : ''}
            <p class="text-gray-600 text-sm">${method.instructions}</p>
        `;
        
        document.getElementById('methodContent').innerHTML = content;
        
        // Scroll to form
        document.getElementById('methodDetails').scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
    
    // Amount selection
    function selectAmount(amount) {
        selectedAmount = amount;
        
        // Update UI
        document.querySelectorAll('.amount-option').forEach(option => {
            option.classList.remove('selected');
        });
        document.getElementById('amount-' + amount).classList.add('selected');
        
        // Update form
        document.getElementById('selectedAmount').value = amount;
        document.getElementById('customAmount').value = amount;
    }
    
    function selectCustomAmount(amount) {
        selectedAmount = parseInt(amount) || 0;
        
        // Remove selection from predefined amounts
        document.querySelectorAll('.amount-option').forEach(option => {
            option.classList.remove('selected');
        });
        
        // Update form
        document.getElementById('selectedAmount').value = selectedAmount;
    }
    
    // Form validation
    document.getElementById('depositForm').addEventListener('submit', function(e) {
        if(!selectedMethod) {
            e.preventDefault();
            showNotification('Please select a payment method', 'error');
            return;
        }
        
        if(!selectedAmount || selectedAmount < 100 || selectedAmount > 50000) {
            e.preventDefault();
            showNotification('Please enter a valid amount between ৳100 and ৳50,000', 'error');
            return;
        }
        
        if(!confirm(`Are you sure you want to deposit ৳${selectedAmount} via ${selectedMethod}?`)) {
            e.preventDefault();
        }
    });
    
    function showNotification(message, type) {
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    // Add animations
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.animate__animated');
        cards.forEach((card, index) => {
            setTimeout(() => {
                card.classList.add('animate__animated', 'animate__fadeIn');
            }, index * 100);
        });
    });
    </script>
</body>
</html>